#ifndef ListAdjGrafo_
#define ListAdjGrafo_

#include <vector>
#include <queue>
#include <list>
#include <stack>
using namespace std;

#include "Vertice.h"
#include "Ramo.h"


template<class TV,class TR>
class ListAdjGrafo
{
private:
	int nvertices;
	int nramos;
	Vertice<TV,TR>* graf;
		
 	Vertice<TV,TR>* encvert_conteudo(const TV& v) const;
	Vertice<TV,TR>* encvert_key(int numvert) const;
	
	void breadthFirstSearch(Vertice<TV,TR> * apinicio) const;
	void depthFirstSearch(Vertice<TV,TR> * apinicio, vector<bool>& vv) const;
	
	void pesquisaCaminhosSimples(Vertice<TV,TR> * apinicio, Vertice<TV,TR> * apfim, vector<bool>& vv, list<Vertice<TV,TR> *>& lv) const;
	void pesquisaCaminhoMinimo(Vertice<TV,TR> * apinicio, Vertice<TV,TR> * apfim) const;
	void pesquisaCaminhoMinimoPesado(Vertice<TV,TR> * apinicio, Vertice<TV,TR> * apfim) const;
		
public:
		ListAdjGrafo();
	ListAdjGrafo(const ListAdjGrafo<TV,TR>& G) ;
	~ListAdjGrafo() ; 
	
	int NumVert() const ;
	int NumRamos() const ;
	
	void juntar_vertice(const TV& vert);
	void juntar_ramo(const TR& rcont, const TV& vorigem, const TV& vdestino);
	
 	int grau_entrada (const TV& vert) const ;
	int grau_saida (const TV& vert) const;
	
	void escreve_grafo() const ;
	
	void visitaLargura(int key) const;
	void visitaProfundidade(int key) const;
	
	void caminhosSimples(int ki, int kf) const;
	void caminhoMinimo(int ki, int kf) const;
	void caminhoMinimoPesado(int ki, int kf) const;
};

template<class TV,class TR>
ListAdjGrafo<TV,TR>::ListAdjGrafo()
{
	nvertices=0;
	nramos=0;
	graf=NULL;
}

template<class TV,class TR>
ListAdjGrafo<TV,TR>::ListAdjGrafo(const ListAdjGrafo<TV,TR>& G)
{
	Vertice<TV,TR>* apv=G.graf ;
	Ramo<TV,TR>* apr;
	
	graf=NULL;
	int numvert=0;
	
	while (apv)		//adiciona os vertices
	{
		numvert++;
		Vertice<TV,TR>* vert=new Vertice<TV,TR>(apv->vconteudo,numvert);
		if (graf == NULL)
			graf = vert ;
		else
		{
			vert->apvertice = graf ; 
			graf = vert ; 
		}
		
		apv = apv->apvertice ; 
	}
	
	nvertices = G.nvertices ;
	
	apv=G.graf;
	while (apv)		 	//adiciona os ramos  
	{
		Vertice<TV,TR>* vorig = encvert_conteudo(apv->vconteudo) ;
		apr = apv->apramo ;		
		while (apr)			 
		{
			Ramo<TV,TR>* ramo = new Ramo<TV,TR>(apr->rconteudo,apr->apv);
			
			if (vorig->apramo == NULL)
		      vorig->apramo=ramo ;
			else
			{
				ramo->apr = vorig->apramo ;
				vorig->apramo = ramo ; 
			}
			apr = apr->apr ;
		} 
		apv = apv->apvertice ; 
	}
	nramos = G.nramos ;
}



template<class TV,class TR>
ListAdjGrafo<TV,TR>::~ListAdjGrafo()
{
	Vertice<TV,TR>* apv=graf;
	Vertice<TV,TR>* tempv ; 
	Ramo<TV,TR>* tempr ;
	Ramo<TV,TR>* temp ;
	
	while (apv)
	{
	  tempr = apv->apramo ; 
	  while (tempr)
	  {
		  temp = tempr ; 
		  tempr = tempr->apr ;
		  delete temp ; 
	  }
	  tempv = apv ; 
		apv = apv->apvertice ;
	  delete tempv ; 
	}
	graf = NULL ;
	nvertices=0 ;
	nramos=0;
}

template<class TV,class TR>
int ListAdjGrafo<TV,TR>::NumVert() const
{
	return nvertices ;
}

template<class TV,class TR>
int ListAdjGrafo<TV,TR>::NumRamos() const
{
	return nramos ;
}


template<class TV,class TR>
Vertice<TV,TR>* ListAdjGrafo<TV,TR>::encvert_conteudo(const TV& v) const
{
	Vertice<TV,TR>* ap=graf;
	
	while (ap != NULL)
	{
		if (ap->vconteudo == v)
			return ap ;
		else 
			ap=ap->apvertice;
	}
	return ap;
}

template<class TV,class TR>
Vertice<TV,TR>* ListAdjGrafo<TV,TR>::encvert_key(int numvert) const
{
	Vertice<TV,TR>* ap=graf;
	
	while (ap != NULL)
	{
		if (ap->key == numvert)
			return ap ;
		else 
			ap=ap->apvertice;
	}
	return ap;
}

template<class TV,class TR>
void ListAdjGrafo<TV,TR>::juntar_vertice(const TV& vert)
{
	if (nvertices == 0)
	{
		nvertices++;
		Vertice<TV,TR>* vertice = new Vertice<TV,TR>(vert,nvertices);
		graf = vertice;
	}
	else
	{
		Vertice<TV,TR>* ap=graf ;
		Vertice<TV,TR>* ant=graf ;
		bool enc=false ; 
		
		while (ap != NULL && !enc)
		{
			if (ap->vconteudo == vert)
				enc=true ;
			else
			{
				ant = ap ; 
				ap=ap->apvertice;
			}
		}
		if (!enc) //vertice nao existe
		{
			nvertices++;	
			Vertice<TV,TR>* vertice=new Vertice<TV,TR>(vert,nvertices);
			ant->apvertice = vertice ;
		}
	}
}


template<class TV,class TR>
void ListAdjGrafo<TV,TR>::juntar_ramo(const TR& rcont, const TV& vorig, const TV& vdest)
{
	Ramo<TV,TR>* tempramo=NULL ; 
	Ramo<TV,TR>* ant;
	Vertice<TV,TR>* vertorig ;
	Vertice<TV,TR>* vertdest=NULL;
 	
	vertorig=encvert_conteudo(vorig);
	if (vertorig == NULL)
	{
		juntar_vertice(vorig) ;
		vertorig=encvert_conteudo(vorig);
	}
	vertdest=encvert_conteudo(vdest);
	if (vertdest == NULL)
	{
		juntar_vertice(vdest) ;
		vertdest=encvert_conteudo(vdest);
	}
	
	tempramo=vertorig->apramo;			//insere no fim da lista de ramos
	ant = tempramo ; 
	while (tempramo != NULL)
	{
		ant = tempramo ; 
		tempramo=tempramo->apr ; 
	}
	if (tempramo == NULL)							 
	{
		tempramo = new Ramo<TV,TR>(rcont,vertdest);
		tempramo->apr= NULL ;
		if (ant)
		  ant->apr = tempramo ;
		else
			vertorig->apramo = tempramo ; 
 		nramos++;
	}
}


template<class TV,class TR>
int ListAdjGrafo<TV,TR>::grau_entrada (const TV& vert) const 
{
	int grau = 0;
	Vertice<TV,TR>* ap=graf;
	Ramo<TV,TR>* ap1;
	
 	while(ap)
	{
		ap1=ap->apramo;
		while(ap1)
		{
			if (vert == ap1->apv->vconteudo) 
			  grau++ ;
			
			ap1=ap1->apr ; 
		}
		ap=ap->apvertice ;
	}
	return grau ;
}


template<class TV,class TR>
int ListAdjGrafo<TV,TR>::grau_saida (const TV& vert) const
{
	Vertice<TV,TR>* ap=encvert_conteudo(vert) ;
	Ramo<TV,TR>* ap1;
	int grau = 0 ;
	
	if (ap->vconteudo == vert)
	{
		ap1=ap->apramo;
		while (ap1)  
		{
			grau++ ;
			ap1=ap1->apr ; 
		}
		ap=ap->apvertice ;
	}
	return grau ;
}

template<class TV,class TR>
void ListAdjGrafo<TV,TR>::escreve_grafo() const 
{
	Vertice<TV,TR>* v=graf;
	Ramo<TV,TR>* r ;
	
	if (v == NULL)
		cout << "Grafo nao definido !" << endl ; 
	else
	{
		cout << "Num vertices " << nvertices  << endl ; 
		cout << "Num ramos " << nramos  << endl ; 
		
		while (v != NULL)
		{
			cout << "O vertice " << v->vconteudo << " liga-se a: " << endl;
 			r=v->apramo;
			while (r)
			{
				cout << r->apv->vconteudo << "  " ;
				cout <<" Conteudo -> "<< r->rconteudo << endl;	
				r=r->apr;
			}   
			cout<<endl;
			v=v->apvertice;
		}
	}
}


// Método de interface para o motor de visita em largura.
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::visitaLargura(int key) const
{
	Vertice<TV,TR> * apinicio = encvert_key(key);
	
	cout << "\n\nVISITA EM LARGURA:\n";
	
	if (apinicio != 0)
		breadthFirstSearch(apinicio);
	
}


// Método motor de visita em largura.
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::breadthFirstSearch(Vertice<TV,TR> * apinicio) const
{
	vector<bool> vv(nvertices+1, false);
	queue<Vertice<TV,TR> *> lv;
	
	vv[apinicio->key] = true;
	cout << "Vertice: " << apinicio->vconteudo << " : " << apinicio->key << endl;
	
	lv.push(apinicio);
	
	// Enquanto houver um vértice por visitar...
	while(!lv.empty())
	{
		apinicio = lv.front();
		lv.pop();
		
		// ... visita todos os vértices adjacentes.
		Ramo<TV,TR> * apramo = apinicio->apramo;
		while (apramo != 0)
		{
			int indice = apramo->apv->key;
			if (vv[indice] == false)
			{
				vv[indice] = true;
				cout << "Vertice: " << apramo->apv->vconteudo << " : " << apramo->apv->key << endl;
				lv.push(apramo->apv);
			}
			apramo = apramo->apr;
		}
	}
}


// Método de interface para o motor de visita em profundidade.
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::visitaProfundidade(int key) const
{
	Vertice<TV,TR> * apinicio = encvert_key(key);
	
		cout << "\n\nVISITA EM PROFUNDIDADE:\n";
	
	if (apinicio != 0)
	{
		vector<bool> vv(nvertices+1, false);  // Vector de vértices visitados.
		
		depthFirstSearch(apinicio, vv);
	}
}


// Método motor da visita em profundidade (recursivo).
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::depthFirstSearch(Vertice<TV,TR> * apinicio, vector<bool>& vv) const
{
	vv[apinicio->key] = true;
	
	cout << "Vertice: " << apinicio->vconteudo << " : " << apinicio->key << endl;
	
	Ramo<TV,TR> * apramo = apinicio->apramo;
	
	// Enquando houver vértices adjacentes...
	while (apramo != 0)
	{
		int indice = apramo->apv->key;
		
		// Se o vértice adjacente não tiver sido previamente visitado...
		if (vv[indice] == false)
			// ... lança o método de visita a partir do vértice adjacente.
			depthFirstSearch(apramo->apv, vv);
		
		// Segue para o próximo vértice adjacente.
		apramo = apramo->apr;
	}
}


// Método de interface para o motor de descoberta de caminhos simples entre
// dois vértices do grafo.
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::caminhosSimples(int ki, int kf) const
{
	Vertice<TV,TR> *apinicio, * apfim;
	
	apinicio = encvert_key(ki);
	apfim = encvert_key(kf);
	
	if (apinicio!=0 && apfim!=0)
	{
		vector<bool> vv(nvertices+1, false);  // Vector de vértices visitados.
		list<Vertice<TV,TR> *>lv;  // Lista dos vértices que compõem o caminho actual.
		
		pesquisaCaminhosSimples(apinicio, apfim, vv, lv);
	}
}


// Motor de descoberta de caminhos simples entre dois vértices do grafo (recursivo).
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::pesquisaCaminhosSimples(Vertice<TV,TR> * apinicio, Vertice<TV,TR> * apfim, vector<bool>& vv, list<Vertice<TV,TR> *>& lv) const
{
	// Este vértice já está visitado. Evita passar de novo por aqui.
	vv[apinicio->key] = true;
	
	lv.push_back(apinicio);  // Insere o vértice actual na lista.
	
	Ramo<TV,TR> * apramo = apinicio->apramo;
	
	// Enquanto houver vértices adjacentes...
	while (apramo!=0)
	{
		int nvdest = apramo->apv->key;
		
		// O vértice adjacente é o destino final? Pode-se comparar o conteudo, a key
		// ou até mesmo o endereço do vértice (experimente).
		if (apramo->apv->vconteudo==apfim->vconteudo)
		{
			lv.push_back(apramo->apv);  // O vértice final é incluido no caminho actual.
			cout << "Caminho simples encontrado: ";
			
			// Escreve o caminho actual na consola.
			typename list<Vertice<TV,TR>*>::iterator it;
			for (it=lv.begin(); it!=lv.end(); it++)
				cout << (*it)->vconteudo << " ";
			
			cout << endl;
			lv.pop_back();  // Retira o vertice destino do caminho actual.
		}
		else
			// Se o vértice adjacente não estiver marcado como visitado, pode seguir-se
			// por aqui a explorar caminhos possíveis.
			if (vv[nvdest] == false)
				// Tenta encontrar um caminho para o destino, a partir deste vértice adjacente.
				pesquisaCaminhosSimples(apramo->apv, apfim, vv, lv);
		
		// Seguir para o próximo vértive adjacente
		apramo = apramo->apr;
	}
	
	// Neste momento já se processaram todos os vértices adjacentes. Antes de
	// terminar o processamento deste vértice...
	lv.pop_back();  // ... retira-se o vértice actual da lista do caminho actual.
	vv[apinicio->key] = false;  // ...  permite-se que no futuro este vértice possa
															// ser utilizado na descoberta de outros caminhos.
}



// Método de interface para o motor de pesquisa de caminho mínimo (não pesado).
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::caminhoMinimo(int ki, int kf) const
{
	Vertice<TV,TR> *apinicio, * apfim;
	
	apinicio = encvert_key(ki);
	apfim = encvert_key(kf);
	
	if (apinicio!=0 && apfim!=0)
	{
		pesquisaCaminhoMinimo(apinicio, apfim);
	}
}


// Método que realiza a pesquisa do caminho mínimo (não-pesado) entre dois
// vértices do grafo.
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::pesquisaCaminhoMinimo(Vertice<TV,TR> * apinicio, Vertice<TV,TR> * apfim) const
{
	vector<int> d(nvertices+1, -1);  // Distâncias mais curtas dos vértices à origem.
																	 // (-1) significa <infinito>: não há distâncias negativas.
	vector<int> p(nvertices+1, 0);  // Vértice antecessor, pelo trajecto mais curto.
	queue<Vertice<TV,TR> *> lv;  // Vértices adjacentes a processar.
	
	d[apinicio->key] = 0;  // A origem está à distância ZERO dela própria.
	
	lv.push(apinicio);
	
	// Enquanto houver vértices alcançáveis para processar...
	while(!lv.empty())
	{
		apinicio = lv.front();
		lv.pop();
		
		Ramo<TV,TR> * apramo = apinicio->apramo;
		// Enquanto houver vértices adjacentes...
		while (apramo != 0)
		{
			int indice = apramo->apv->key;
			
			// Se ainda não se passou por este vértice adjacente...
			if (d[indice] == -1)
			{
				d[indice] = d[apinicio->key]+1;  // Marcar a distância.
				p[apramo->apv->key] = apinicio->key;  // Marcar o antecessor.
				lv.push(apramo->apv);  // Este vértice tem de ser processado (ver os seus adjacentes).
			}
			
			// Seguir para o próximo vértice adjacente.
			apramo = apramo->apr;
		}
	}
	
	// Já se visitaram todos os vértices alcançáveis...
	
	// Estabelecer o caminho mínimo entre os dois vértices, a partir do destino.
	if (d[apfim->key] >= 0)
	{
		stack<Vertice<TV,TR> *> sp; // Pilha para guardar o caminho (ultimo->primeiro).
		int indice = apfim->key;
		sp.push(encvert_key(indice));
		
		while (p[indice]!=0)
		{
			indice = p[indice];
			sp.push(encvert_key(indice));
		}
		
		// Escrever o caminho (primeiro->último).
		cout << "Caminho mínimo: ";
		while (!sp.empty())
		{
			cout << sp.top()->vconteudo << " ";
			sp.pop();
		}
		cout << "\n\n";
	}	
}


// Método de interface para o motor de pesquisa do caminho mínimo pesado (Dijkstra).
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::caminhoMinimoPesado(int ki, int kf) const
{
	Vertice<TV,TR> *apinicio, * apfim;
	
	apinicio = encvert_key(ki);
	apfim = encvert_key(kf);
	
	if (apinicio!=0 && apfim!=0)
	{
		pesquisaCaminhoMinimoPesado(apinicio, apfim);
	}
}


// Motor de pesquisa de caminho mínimo pesado(Dijkstra).
template<class TV,class TR>
void ListAdjGrafo<TV,TR>::pesquisaCaminhoMinimoPesado(Vertice<TV,TR> * apinicio, Vertice<TV,TR> * apfim) const
{
	vector<int> d(nvertices+1, -1);  // Distãncia mais curta à origem.
	vector<int> p(nvertices+1, 0);  // Vértice antecessor, pelo caminho mais curto.
	vector<bool> proc(nvertices+1, false);  // Vértice processado.
	
	d[apinicio->key] = 0;
	
	Vertice<TV,TR> * apactual = apinicio;
	
	// Enquanto houver vértices alcançáveis...
	while(apactual!=0)
	{
		proc[apactual->key] = true;  // Vértice processado.
		
		Ramo<TV,TR> * apramo = apactual->apramo;
		// Enquanto houver vértices adjacentes...
		while (apramo != 0)
		{
			int indice = apramo->apv->key;
			// Se o vértice ainda não tiver sido considerado processado...
			if (proc[indice]==false)
			{
				// Se for encontrada uma distância mais curta (-1 representa infinito)...
				if (d[indice]==-1 || (d[indice]>(d[apactual->key]+apramo->rconteudo)))
				{
					d[indice] = d[apactual->key] + apramo->rconteudo;
					p[indice] = apactual->key;
				}
			}
			// Para o próximo ramo adjacente.
			apramo = apramo->apr;
		}
		
		// Qual o nó por processar com menor distância à origem?
		int imin = 0;
		int dmin = 0;
		for(int i=1; i<=proc.size(); i++)
		{
			if (d[i]>0 && proc[i]==false)
				if (dmin==0 || dmin>d[i])
				{
					dmin = d[i];
					imin = i;
				}
		}
			
		// Segue a partir do vértice mais próximo da origem, por processar.
		apactual = encvert_key(imin);
		
	}
	
	// Já se visitaram todos os vértices alcançáveis...
	
	// Estabelecer o caminho mínimo entre os dois vértices, a partir do destino.
	if (d[apfim->key] >= 0)
	{
		stack<Vertice<TV,TR> *> sp;  // Pilha para guarda o caminho mínimo (utlimo->primeiro)
		int indice = apfim->key;
		sp.push(encvert_key(indice));
		
		while (p[indice]!=0)
		{
			indice = p[indice];
			sp.push(encvert_key(indice));
		}
		
		cout << "Caminho mínimo: ";
		// Escreve o caminho mínimo (primeiro->último).
		while (!sp.empty())
		{
			cout << sp.top()->vconteudo << " ";
			sp.pop();
		}
		cout << "\n\n";
	}	
}


#endif